/*
  C program that demonstrates the use of OpenAL together
  with OpenGL.

  Note.
  Because of the use of the three dimensional array ctrlpoints,
  it's actually C++ and should therefore be compiled with a C++
  compiler such as g++. Apart from this it's ANSI C.
*/

#include <GL/glut.h>
#include <AL/al.h>
#include <AL/alut.h>
#include <stdio.h>
#include <stdlib.h>

/* Constants used by OpenAL */
#define NUM_BUFFERS 2
#define NUM_SOURCES 2
#define NUM_ENVIRONMENTS 1

/* My own constants used in the GLUT menu */
#define SHOW_DIRECTIONS 0
#define SHOW_CONTROLPOINTS 11
#define ANIMATE 12
#define FULL_SCREEN 100
#define QUIT_SIMPLE 1

/* Rotation variables */
GLfloat vx,vz;
GLfloat lastx,lasty;
GLint moving; 

/* The z coordinate for the bass element in the speaker */
GLfloat bassZ = 0.0;

/* Light and material variables */
GLfloat light0_position[] = { -1.0, 1.0, 1.0, 0.0 };
GLfloat light0_ambient[] = { 1.0, 1.0, 1.0, 0.0 };
GLfloat light0_diffuse[] = { 1.0, 1.0, 1.0, 0.0 };
GLfloat light0_specular[] = { 1.0, 1.0, 1.0, 0.0 };
GLfloat spot_direction[] = { 1.0, 0.0, 0.0 };

GLfloat mat_ambient[] = { 0.0, 0.0, 1.0, 1.0 };
GLfloat mat_diffuse[] = { 0.5, 0.0, 0.5, 1.0 };
GLfloat mat_specular[] = { 1.0, 0.8, 0.2, 1.0 };
GLfloat mat_shininess[] = { 40.0 };

/* 
   The front of the speakers.

   Bezier surface control points, [ v ][ u ][ xyz ] 
   16 control points in 4x4 field, 
   the first block of coordinates is the bottom line.
*/
GLfloat ctrlpoints[ 4 ][ 4 ][ 3 ] = 
{
   { 
      { 0.0f, 0.0f, 0.0f }, { 0.25f, 0.0f, 0.0f }, 
      { 0.5f, 0.0f, 0.0f }, { 1.0f, 0.0f, 0.0f }
   }, 
   { 
      { 0.0f, 0.25f, 0.0f }, { 0.25f, 0.25f, bassZ }, 
      { 0.5f, 0.25f, bassZ }, { 1.0f, 0.25f, 0.0f }
   }, 
   {
      { 0.0f, 0.5f, 0.0f }, { 0.25f, 0.5f, bassZ }, 
      { 0.5f, 0.5f, bassZ }, { 1.0f, 0.5f, 0.0f }
   }, 
   { 
      { 0.0f, 1.0f, 0.0f }, { 0.25f, 1.0f, 0.0f }, 
      { 0.5f, 1.0f, 0.0f }, { 1.0f, 1.0f, 0.0f }
   }
};

int GLwin;
int showDirections = 0;
int showControlpoints = 0;
int animate = 0;
int musicStarted = 0;
float xDirection = 1.0;
float yDirection = 1.0;
float zDirection = 1.0;

/* OpenAL variables */
ALuint buffer[ NUM_BUFFERS ];
ALuint source[ NUM_SOURCES ];
ALuint environment[ NUM_ENVIRONMENTS ];

ALsizei size, freq, bits;
ALenum format;
ALvoid *data;
ALboolean err;

ALfloat source0Position[] = { -2.0, 0.0, -1.0 };
ALfloat source0Velocity[] = { 0.0, 0.0, 0.0 };
ALfloat source0Direction[] = { 0.0, 0.0, 1.0 };

ALfloat source1Position[] = { 2.0, 0.0, -1.0 };
ALfloat source1Velocity[] = { 0.0, 0.0, 0.0 };
ALfloat source1Direction[] = { 0.0, 0.0, 1.0 };

ALfloat listenerPosition[] = { 0.0, 0.5, 6.0 };
ALfloat listenerVelocity[] = { 0.0, 0.0, 0.0 };
ALfloat listenerOrientation[] = { 0.0, 0.0, 2.0,
                                  0.0, 1.0, 0.0 };

/* 
   Pointers to the two bitmaps used for texture mapping.
   Instead of loading the textures anew each time they're
   used, their contents are put into memory and referrenced
   to by the pointers below.
*/
GLubyte *floor_bitmap;
GLubyte *speaker_bitmap;

/*
  Mettod for loading the texures used for texture mapping
  the front of the speakers and the floor.
 */
void loadTextures( void )
{
   FILE *file;

   /* The floor bitmap */
   if( ( file = fopen( "floor.rgb", "rb" ) ) == NULL )
   {
      printf( "File not found : %s\n", "floor.rgb" );
      exit( 1 );
   }

   floor_bitmap = ( GLubyte * ) malloc ( 64 * 64 * 7 * ( sizeof( GLubyte ) ) );

   if( floor_bitmap == NULL )
   {
      printf( "Cannot allocate memory for texture\n" );
      fclose( file );
      exit( 1 );
   }

   fread( floor_bitmap , 64 * 64 * 7, 1 , file );
   fclose( file );

   /* The speaker bitmap */
   if( ( file = fopen( "openal_speaker.rgb", "rb" ) ) == NULL )
   {
      printf( "File not found : %s\n", "openal_speaker.rgb" );
      exit( 1 );
   }

   speaker_bitmap = ( GLubyte * ) malloc ( 64 * 64 * 6 * ( sizeof( GLubyte ) ) );

   if( floor_bitmap == NULL )
   {
      printf( "Cannot allocate memory for texture\n" );
      fclose( file );
      exit( 1 );
   }

   fread( speaker_bitmap , 64 * 64 * 6, 1 , file );
   fclose( file );

   /* GL_MODULATE, texture and polygon share properties. */
   glTexEnvf( GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_REPLACE );
}

/*
  Method used each time the floor is to be rendered.
  Creates mipmaps to preserve image quality when moving 
  closer/farer away from the object. The method uses
  the pointer floor_bitmap, in stead of loading
  the bitmap each time.
*/
void getFloorTexture( void )
{
   gluBuild2DMipmaps( GL_TEXTURE_2D, GL_RGB, 64, 64,
                      GL_RGB, GL_UNSIGNED_BYTE, floor_bitmap );

}

/*
  Same as getFloorTexture(), but this time
  it is for the front of the speakers.
*/
void getSpeakerTexture( void )
{
   gluBuild2DMipmaps( GL_TEXTURE_2D, GL_RGB, 64, 64,
                      GL_RGB, GL_UNSIGNED_BYTE, speaker_bitmap );

}

/*
  Method used for drawing the speakers,
  the front is a texture mapped bezier
  surface, whereas the left, back, right and
  top are just fille polygons.
 */
void drawSpeaker( void )
{

   int u, v;

   getSpeakerTexture();

   glEnable( GL_TEXTURE_2D );

   GLfloat txpts[ 2 ][ 2 ][ 2 ] =
      {
         {
            { 0.0, 1.0 }, { 1.0, 1.0 }
         },
         {
            { 0.0, 0.0 }, { 1.0, 0.0 }
         }
  };

   glTexEnvi( GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE,GL_DECAL );   
   glTexParameteri( GL_TEXTURE_2D,GL_TEXTURE_MIN_FILTER,GL_NEAREST );

   /*
     The picture used for the floor is really small, only 16x16 pixels.
     In order to get it "smoothed", I commented out the line below.
     This way, the texture has been stretched, but the edges between each
     pixel is "smoothed".

     glTexParameteri( GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_NEAREST );
   */

   glTexParameteri( GL_TEXTURE_2D,GL_TEXTURE_WRAP_S,GL_REPEAT );
   glTexParameteri( GL_TEXTURE_2D,GL_TEXTURE_WRAP_T,GL_REPEAT );

   glEnable( GL_MAP2_VERTEX_3 );
   glEnable( GL_MAP2_TEXTURE_COORD_1 );
   glEnable( GL_MAP2_TEXTURE_COORD_2 );   

   glMap2f( GL_MAP2_VERTEX_3,
            0.0f, 1.0f, 3, 4, 0.0f, 1.0f, 12, 4,
           &ctrlpoints[ 0 ][ 0 ][ 0 ] );

   glMap2f( GL_MAP2_TEXTURE_COORD_2, 
            0, 1, 2, 2, 0, 1, 4, 2,
            &txpts[ 0 ][ 0 ][ 0 ] );   
   
   glMapGrid2f( 20,0.0, 1.0, 20, 0.0, 1.0 );   
  
   glEvalMesh2( GL_FILL, 0, 20, 0, 20 );

   glDisable( GL_TEXTURE_2D );
   glDisable( GL_MAP2_VERTEX_3 );
   glDisable( GL_MAP2_TEXTURE_COORD_1 );
   glDisable( GL_MAP2_TEXTURE_COORD_2 );   

   if( showControlpoints == 1 )
   {
      glColor3f( 0.0, 0.0, 0.5 );
 
      for( u=0; u<4 ; u++ )
      {
         glBegin( GL_LINE_STRIP );
            for( v=0; v < 4; v++ )
            {
               glVertex3fv( ctrlpoints[ v ][ u ] );
            }
         glEnd();
      }
      for( v = 0; v < 4; v++ )
      {
         glBegin( GL_LINE_STRIP );	
            for( u=0; u < 4; u++ )
            {
               glVertex3fv( ctrlpoints[ v ][ u ] );
            }
         glEnd();
      }
   }

   /*  Left, back, right and top  */
   glColor3f( 0.23, 0.08, 0.05 );

   glBegin( GL_POLYGON );
      glVertex3f( 0.0, 0.0, 0.0 ); 
      glVertex3f( 0.0, 0.0, -1.0 );
      glVertex3f( 0.0, 1.0, -1.0 );
      glVertex3f( 0.0, 1.0, 0.0 );
   glEnd();
   glBegin( GL_POLYGON );
      glVertex3f( 0.0, 0.0, -1.0 );
      glVertex3f( 1.0, 0.0, -1.0 );
      glVertex3f( 1.0, 1.0, -1.0 );
      glVertex3f( 0.0, 1.0, -1.0 );
   glEnd();
   glBegin( GL_POLYGON );
      glVertex3f( 1.0, 0.0, -1.0 );
      glVertex3f( 1.0, 0.0, 0.0 );
      glVertex3f( 1.0, 1.0, 0.0 );
      glVertex3f( 1.0, 1.0, -1.0 );
   glEnd();
   glBegin( GL_POLYGON );
      glVertex3f( 0.0, 1.0, 0.0 );
      glVertex3f( 0.0, 1.0, -1.0 );
      glVertex3f( 1.0, 1.0, -1.0 );
      glVertex3f( 1.0, 1.0, 0.0 );
   glEnd();
}

/*
  Method used for drawing the floor.
 */
void drawFloor( void )
{
   glPixelStorei( GL_UNPACK_ALIGNMENT, 1 );
   getFloorTexture();
   glColor3f( 1.0, 1.0, 1.0 );

   glEnable( GL_TEXTURE_2D );
      glBegin( GL_POLYGON );
         glNormal3f( 0.0, 1.0, 0.0 );
         glTexCoord2f( 1.0, 0.0 ); glVertex3f( -12.0, 0.0, -12.0 );
         glTexCoord2f( 1.0, 1.0 ); glVertex3f( 12.0, 0.0, -12.0 );
         glTexCoord2f( 0.0, 1.0 ); glVertex3f( 12.0, 0.0, 12.0 );
         glTexCoord2f( 0.0, 0.0 ); glVertex3f( -12.0, 0.0, 12.0 );
      glEnd();
   glDisable( GL_TEXTURE_2D );
}

/*
  Setting up the sound.
  All of the lines below are pretty much OpenAL 
  specific.
 */
void initSound( void )
{
   alListenerfv( AL_POSITION, listenerPosition );
   alListenerfv( AL_VELOCITY, listenerVelocity );
   alListenerfv( AL_ORIENTATION, listenerOrientation );

   alGetError();
   alGenBuffers( NUM_BUFFERS, buffer );
   
   if( alGetError() != AL_NO_ERROR )
   {
      printf( "Error creating buffers." );
      exit( 1 );
   }

   err = alutLoadWAV( "art.wav", &data, &format, &size, &bits, &freq );

	if( err == AL_FALSE ) 
   {
		fprintf( stderr, "Could not include %s\n", "b.wav" );
		exit( 1 );
	}

   alBufferData( buffer[ 0 ], format, data, size, freq );
   alBufferData( buffer[ 1 ], format, data, size, freq );

   free( data ); 

   alGetError();
   alGenSources( NUM_SOURCES, source );

   if( alGetError() != AL_NO_ERROR )
   {
      printf( "Error creating source" );
      exit( 2 );
   }

   alSourcef( source[ 0 ], AL_PITCH, 1.0f );
   alSourcef( source[ 0 ], AL_GAIN, 1.0f );
   alSourcefv( source[ 0 ], AL_POSITION, source0Position );
   alSourcefv( source[ 0 ], AL_VELOCITY, source0Velocity );
   alSourcei( source[ 0 ], AL_BUFFER, buffer[ 0 ] );
   alSourcei( source[ 0 ], AL_LOOPING, AL_TRUE );
   alSourcefv( source[ 0 ], AL_DIRECTION, source0Direction );

   alSourcef( source[ 1 ], AL_PITCH, 1.0f );
   alSourcef( source[ 1 ], AL_GAIN, 1.0f );
   alSourcefv( source[ 1 ], AL_POSITION, source1Position );
   alSourcefv( source[ 1 ], AL_VELOCITY, source1Velocity );
   alSourcei( source[ 1 ], AL_BUFFER, buffer[ 1 ] );
   alSourcei( source[ 1 ], AL_LOOPING, AL_TRUE );
   alSourcefv( source[ 1 ], AL_DIRECTION, source1Direction );
}

void myInit( void )
{
   glClearColor( 0.0, 0.0, 0.0, 1.0 );

   glShadeModel( GL_SMOOTH );

   /* Light */
   glEnable( GL_NORMALIZE );
   glEnable( GL_LIGHTING );
   glEnable( GL_LIGHT0 );

   glLightfv( GL_LIGHT0, GL_POSITION, light0_position );
   glLightfv( GL_LIGHT0, GL_AMBIENT, light0_ambient );
   glLightfv( GL_LIGHT0, GL_DIFFUSE, light0_diffuse );
   glLightfv( GL_LIGHT0, GL_SPECULAR, light0_specular );

   glLightf( GL_LIGHT0, GL_SPOT_CUTOFF, 90.0 );
   glLightfv( GL_LIGHT0, GL_SPOT_DIRECTION, spot_direction );

   glMaterialfv( GL_FRONT, GL_AMBIENT, mat_ambient );
   glMaterialfv( GL_FRONT, GL_DIFFUSE, mat_diffuse );
   glMaterialfv( GL_FRONT, GL_SPECULAR, mat_specular );
   glMaterialfv( GL_FRONT, GL_SHININESS, mat_shininess );

   initSound();

   loadTextures();

   glEnable( GL_DEPTH_TEST );
   glEnable ( GL_COLOR_MATERIAL );

}

/*
  Method that does the real rendering of the scene,
  called "all the time".
 */
void myDisplay( void )
{
   glClear( GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT );

   glPushMatrix();
      glMatrixMode( GL_MODELVIEW );
      glLoadIdentity();
      gluLookAt( listenerPosition[ 0 ], listenerPosition[ 1 ], listenerPosition[ 2 ],
                 0.0, 0.0, 0.0,
                 0.0, 1.0, 0.0 );

      /*
        glRotatef( 20.0, 1.0, 1.0, 0.0 );
      */
      glRotatef( vx, 1.0f, 0.0f, 0.0f );
      glRotatef( vz, 0.0f, 0.0f, 1.0f );

      glPushMatrix();
         drawFloor();
      glPopMatrix();

      glPushMatrix();
         glScalef( 1.0, 2.0, 1.0 );
         
         if( showDirections == 1 )
         {
            glColor3f( 1.0, 1.0, 1.0 );
            glBegin( GL_LINES );
               glVertex3f( source0Position[ 0 ], source0Position[ 1 ], source0Position[ 2 ] );
               glVertex3f( source0Direction[ 0 ], source0Direction[ 1 ], source0Direction[ 2 ] );
            glEnd();
         }

         glTranslatef( source0Position[ 0 ], source0Position[ 1 ], source0Position[ 2 ] );
         glColor3f( 1.0, 0.0, 0.0 );
         drawSpeaker();
      glPopMatrix();

      glPushMatrix();
         glScalef( 1.0, 2.0, 1.0 );

         if( showDirections == 1 )
         {
            glColor3f( 1.0, 1.0, 1.0 );
            glBegin( GL_LINES );
               glVertex3f( source1Position[ 0 ], source1Position[ 1 ], source1Position[ 2 ] );
               glVertex3f( source1Direction[ 0 ], source1Direction[ 1 ], source1Direction[ 2 ] );
            glEnd();
         }

         glTranslatef( source1Position[ 0 ], source1Position[ 1 ], source1Position[ 2 ] );
         glColor3f( 0.0, 1.0, 0.0 );
         drawSpeaker();
      glPopMatrix();

      glPushMatrix();
 
         if( showDirections == 1 )
         {
            glColor3f( 1.0, 1.0, 1.0 );
            glBegin( GL_LINES );
               glVertex3f( listenerPosition[ 0 ], listenerPosition[ 1 ], listenerPosition[ 2 ] );
               glVertex3f( listenerOrientation[ 0 ], listenerOrientation[ 1 ], listenerOrientation[ 2 ] );
            glEnd();
         }
   
         glTranslatef( listenerPosition[ 0 ], listenerPosition[ 1 ], listenerPosition[ 2 ] );
         glutSolidSphere( 0.5, 20, 16 );
      glPopMatrix();
   glPopMatrix();
   
   glutSwapBuffers();

   /*
     Used for debugging, listing where we are in 
     3D :-)

   printf( "x: %f y: %f z: %f %s", listenerPosition[ 0 ], 
           listenerPosition[ 1 ], listenerPosition[ 2 ], "\n" );
   */
}

/*
  Called when the window is resized, ie. maximized.
 */
void myReshape( int w, int h )
{
   glViewport( 0, 0, ( GLsizei )w, ( GLsizei )h );

   glMatrixMode( GL_PROJECTION );
   glLoadIdentity();
   gluPerspective( 60.0, ( GLfloat )w/( GLfloat )h, 1.0, 30.0 );
 
   glPushMatrix();
      glMatrixMode( GL_MODELVIEW );
      glLoadIdentity();
      glTranslatef( 0.0, 0.0, -6.6 );
   glPopMatrix();
}

/*
  Handles all keyboard interaction.
 */
void myKeyboard( unsigned char key, int x, int y )
{
   switch( key )
   {
   case '1':
      alSourcePlay( source[ 0 ] );
      alSourcePlay( source[ 1 ] );
      musicStarted = 1;
      break;
   case '2':
      alSourceStop( source[ 0 ] );
      alSourceStop( source[ 1 ] );
      musicStarted = 0;
      break;
   case 'a':                                          
      listenerPosition[ 0 ] -= 0.1;
      alListenerfv( AL_POSITION, listenerPosition );
      break;
   case 's':
      listenerPosition[ 0 ] += 0.1;
      alListenerfv( AL_POSITION, listenerPosition );
      break;
   case 'q':
      listenerPosition[ 2 ] -= 0.1;
      alListenerfv( AL_POSITION, listenerPosition );
      break;
   case 'z':
      listenerPosition[ 2 ] += 0.1;
      alListenerfv( AL_POSITION, listenerPosition );
      break;
   case 'd':
      if( showDirections == 0 )
      {
         showDirections = 1;
      }
      else 
      {
         showDirections = 0;
      }

      break;
   /* Hittng the space bar causes the view to be levelled. */
   case 32: 
      vx = 0.0;
      vz = 0.0;
      break;
   case 27:
      alSourceStop( source[ 0 ] );
      alSourceStop( source[ 1 ] );
      alutExit();
      glutDestroyWindow( GLwin );
      exit( 0 );
      break;
   }

   glutPostRedisplay();
}

/*
  Clicking and dragging with the mouse goes in here.
 */
void myMouse( int button, int state, int x, int y )
{
   if( button == GLUT_LEFT_BUTTON && state == GLUT_UP )
   {
      glutPostRedisplay();
   }

   if( button == GLUT_LEFT_BUTTON && state == GLUT_DOWN )
   {
      moving = 1;
      lastx = x;
      lasty = y;
   }
   else if( button == GLUT_LEFT_BUTTON && state == GLUT_UP )
   {
      moving = 0;
   }
}

/*
  Algorithm for draggin the object "the screne in this case"
  is taken from Brre Stenseth, http://www.hiof.no/~borres/gb/
 */
void myMotion( int x,int y )
{
   if( moving )
   {
      vx += 90.0f * ( ( 1.0f * ( x - lastx ) ) / ( 1.0f * glutGet( GLUT_WINDOW_WIDTH ) ) );
      vz += 90.0f * ( ( 1.0f * ( y - lasty ) ) / ( 1.0f * glutGet( GLUT_WINDOW_HEIGHT ) ) );

      lastx = x;
      lasty = y;

      glutPostRedisplay();
   }
} 

/*
  Gives us a simple GLUT menu 
  when clicking on the right mouse button.
 */
void myMenuHandler( int item )
{
    switch( item ) 
    {

    case SHOW_DIRECTIONS:

       if( showDirections == 0 )
       {
          showDirections = 1;
       }
       else 
       {
          showDirections = 0;
       }

       glutPostRedisplay();
       break;

    case QUIT_SIMPLE:

       alSourceStop( source[ 0 ] );
       alSourceStop( source[ 1 ] );
       alutExit();
       glutDestroyWindow( GLwin );

       free( floor_bitmap );
       free( speaker_bitmap );

       exit( 0 );
       break;

    case SHOW_CONTROLPOINTS:

       if( showControlpoints == 1 )
       {
          showControlpoints = 0;
       }
       else 
       {
          showControlpoints = 1;
       }

       break;

    case ANIMATE:
       if( animate == 1 )
       {
          animate = 0;}
       else 
       {
          animate = 1;
       }
       break;

    case FULL_SCREEN:
       glutFullScreen();
       glutPostRedisplay();
       break;
    }
} 

void myIdleFunction( void )
{
   /*
     Checks that the viewer can "see" the front of the speaker, 
     according to where we are in the x, y, z space.
     If we cannot "see" the front of the speaker, it doesn't
     move its bass elements.
  
   */
   if( listenerPosition[ 2 ] > -2.0 && listenerPosition[ 2 ] < 20.0 && musicStarted == 1 )
   {
      if( bassZ < 0.0 )
      {
         bassZ = 0.4;
      }
      else 
      {
         bassZ -= 0.01;
      }

      ctrlpoints[ 1 ][ 1 ][ 2 ] =
         ctrlpoints[ 1 ][ 2 ][ 2 ] =
         ctrlpoints[ 2 ][ 1 ][ 2 ] =
         ctrlpoints[ 2 ][ 2 ][ 2 ] = bassZ;

      glutPostRedisplay();
   }

   /* 
      Moves the camera ( and listener ) around the scene,
      demonstrating how the sound changes as you go from
      right to left, and back and forth.
    */
   if( animate == 1 )
   {
      if( listenerPosition[ 0 ] > 12.0 )
      {
         xDirection = -1.0;
      }
      else if( listenerPosition[ 0 ] < -12.0 )
      {
         xDirection = + 1.0;
      }

      if( listenerPosition[ 2 ] > 12.0 )
      {
         zDirection = -1.0;
      }
      else if( listenerPosition[ 2 ] < -12.0 )
      {
         zDirection = 1.0;
      }

      listenerPosition[ 0 ] += ( xDirection / 100 );
      listenerPosition[ 2 ] += ( zDirection / 100 );
      alListenerfv( AL_POSITION, listenerPosition );

      glutPostRedisplay();
   }
}

/*
  Standard main method.
  Clears all buffers, sets up a window 
  ( or actually GLUT does this for us ),
  runs my own init method, called myInit(),
  sets up the menu, and enters the famous
  GLUT eternal main loop.
 */
int main( int argc, char** argv )
{
   glutInit( &argc, argv );
   glutInitDisplayMode( GLUT_DOUBLE | GLUT_RGB| GLUT_DEPTH );
   glutInitWindowSize( 500, 500 );

   alutInit( &argc, argv );
   GLwin = glutCreateWindow( "OpenGL & OpenAL" );
  
   myInit();
   
   glutDisplayFunc( myDisplay );
   glutKeyboardFunc( myKeyboard );
   glutMouseFunc( myMouse );
   glutMotionFunc( myMotion );
   glutReshapeFunc( myReshape );
   glutIdleFunc( myIdleFunction );

   glutCreateMenu( myMenuHandler );
   glutAddMenuEntry( "Vis retninger", SHOW_DIRECTIONS );
   glutAddMenuEntry( "Vis kontrollpunkter", SHOW_CONTROLPOINTS );
   glutAddMenuEntry( "Animasjon", ANIMATE );
   glutAddMenuEntry( "Full skjerm", FULL_SCREEN );
   glutAddMenuEntry( "Avslutt", QUIT_SIMPLE );
   glutAttachMenu( GLUT_RIGHT_BUTTON );     

   glutMainLoop();

   return 0;
}
